/**
  *******************************************************************************
  * 
  * @file    hal_i2c.c
  * @brief   ES7P203 i2c module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The I2c driver can be used as follows:
    (+) Initialize the I2c registers in slave mode by using i2c_init().
    (+) Transmit in slave mode an amount of data using i2c_slave_send()
    (+) Receive in slave mode an amount of data using i2c_slave_recv()

  @endverbatim
  *********************************************************************************
  */

#include <hic.h>
#include "hal_base.h"
#include "hal_i2c.h"
#include "string.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup I2C i2c
  * @brief i2c module driver
  * @{
  */

/**
  * @defgroup I2c_Public_Functions i2c Public Functions
  * @brief    Initizlization and Data Transfers Functions
@verbatim
=================================================================================
		##### I2C Public Functions #####
=================================================================================
  [..]  The I2c driver can be used as follows:
    (+) Initialize the I2c registers in slave mode by using i2c_init().
    (+) Transmit in slave mode an amount of data using i2c_slave_send()
    (+) Receive in slave mode an amount of data using i2c_slave_recv()
@endverbatim
  * @{
  */

/**
  * @brief  reset i2c mode
  * @param  hperh: i2c slave mode address.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t i2c_reset(i2c_handle_t *hperh)
{
	I2CRST = 1;
	delay(1);

	if (I2CRST)
		return ERROR;

	memset(hperh, 0, sizeof(hperh));

	return OK;	
}

/**
  * @brief  Initialze the I2c accroding to the specified parameters
  *         in the i2c_init_t and initialize the associated handle.
  * @param  hperh: i2c slave mode address.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t i2c_init(i2c_handle_t *hperh)
{

	I2CC   = 0xE0;
	I2CSA  = hperh->addr;
	I2CIEC = 0x43;
	I2CIFC = 0;

	I2CIE = 1;
	I2CEN = 1;

	return OK;
}

/**
  * @brief  I2c irq handler function.
  * @param  hperh: Pointer of i2c handle structure.
  * @retval None
  */
void irq_i2c_handle(i2c_handle_t *hperh)
{
	if ((I2CIE == 1) && (I2CIF == 1)) {
		if ((I2CSRIE == 1) && (I2CSRIF == 1)) {
			I2CSRIF = 0;

			if (I2CRW == 1) {
				I2CRBIE = 0;
				I2CTBIE = 1;
				I2CCSE  = 1;
			}
			else {
				I2CRBIE = 1;
				I2CTBIE = 0;
			}
		}

		if ((I2CTBIE == 1) && (I2CTBIF == 1)) {
			if (hperh->count > 0) {
				I2CTB = *hperh->buf;
				hperh->count--;
				hperh->buf++;

				if (hperh->count == 0)
					I2CTBIE = 0;
			}
		}

		if ((I2CRBIE == 1) && (I2CRBIF == 1)) {
			if (hperh->count > 0) {
				*(hperh->buf) = I2CRB;
				hperh->count--;
				hperh->buf++;
			}
		}

		if ((I2CSPIE == 1) && (I2CSPIF == 1)) {
			I2CSPIF = 0;
			I2CRST  = 1;
			while (I2CRST) {};
			I2CC    = 0xE1;
			I2CIEC  = 0x43;
		}

		if ((I2CNAIE == 1) && (I2CNAIF == 1)) {
			I2CNAIF = 0;
		}

		I2CIF = 0;
	}
}

/**
  * @brief  Transmits in salve mode an amount of data.
  * @param  hperh: Pointer of i2c handle structure.
  * @param  buf: Pointer of data buff want transmit.
  * @param  size: length of transmit data buf.
  * @retval Status, see @ref hal_status_t.
  */
 hal_status_t i2c_tx_by_irq(i2c_handle_t *hperh, uint8_t *buf, uint8_t size)
{	
	if ((buf == NULL) || (size == 0))
		return ERROR;

	hperh->buf   = buf;
	hperh->size  = size;
	hperh->count = size;

	return OK;
}

/**
  * @brief  Receive in salve mode an amount of data.
  * @param  hperh: Pointer of i2c handle structure.
  * @param  buf: Pointer of data buff want transmit.
  * @param  size: length of transmit data buf.
  * @retval Status, see @ref hal_status_t.
  */
 hal_status_t i2c_rx_by_irq(i2c_handle_t *hperh, uint8_t *buf, uint8_t size)
{	
	if ((buf == NULL) || (size == 0))
		return ERROR;

	hperh->buf   = buf;
	hperh->size  = size;
	hperh->count = size;

	return OK;
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
